#!/usr/bin/env bash
set -euo pipefail

COLOR_BLUE="\033[1;34m"
COLOR_GREEN="\033[1;32m"
COLOR_YELLOW="\033[1;33m"
COLOR_RED="\033[1;31m"
COLOR_RESET="\033[0m"

info(){ echo -e "${COLOR_BLUE}➤${COLOR_RESET} $*"; }
ok(){ echo -e "${COLOR_GREEN}✓${COLOR_RESET} $*"; }
warn(){ echo -e "${COLOR_YELLOW}!${COLOR_RESET} $*"; }
err(){ echo -e "${COLOR_RED}✗${COLOR_RESET} $*"; }

require_cmd(){ command -v "$1" >/dev/null 2>&1; }

# Ensure sudo exists (most Arch installs have it)
if ! require_cmd sudo; then
  err "sudo n'est pas disponible. Exécute ce script en root ou installe sudo."
  exit 1
fi
info "Activation du dépôt multilib…"

if pacman-conf --repo-list | grep -qx "multilib"; then
  ok "multilib est déjà activé dans pacman.conf."
  exit 0
fi

conf="/etc/pacman.conf"
backup="/etc/pacman.conf.bak.$(date +%Y%m%d-%H%M%S)"
info "Sauvegarde de pacman.conf -> ${backup}"
sudo cp -a "$conf" "$backup"

# Décommente proprement la section [multilib] (ligne d'en-tête + Include)
# On remplace les occurrences exactes commentées par leurs équivalents non commentés.
sudo awk '
  BEGIN { in_multi=0 }
  {
    if ($0 ~ /^#\s*\[multilib\]\s*$/) { gsub(/^#\s*/, "", $0); in_multi=1; print; next }
    if (in_multi==1 && $0 ~ /^#\s*Include\s*=\s*\/etc\/pacman\.d\/mirrorlist\s*$/) {
      gsub(/^#\s*/, "", $0); in_multi=0; print; next
    }
    print
  }
' "$conf" | sudo tee /tmp/pacman.conf.new >/dev/null

# Vérifie si la modif a ajouté la section ; sinon, essaie une autre approche (section manquante)
if ! grep -q "^\[multilib\]" /tmp/pacman.conf.new; then
  warn "Section [multilib] commentée introuvable, on l’ajoute à la fin du fichier."
  printf "\n[multilib]\nInclude = /etc/pacman.d/mirrorlist\n" | sudo tee -a /tmp/pacman.conf.new >/dev/null
fi

sudo mv /tmp/pacman.conf.new "$conf"
ok "multilib activé."
info "Mise à jour des bases de paquets…"
sudo pacman -Syy --noconfirm
ok "Terminé."
