#!/usr/bin/env bash
set -euo pipefail

COLOR_BLUE="\033[1;34m"
COLOR_GREEN="\033[1;32m"
COLOR_YELLOW="\033[1;33m"
COLOR_RED="\033[1;31m"
COLOR_RESET="\033[0m"

info(){ echo -e "${COLOR_BLUE}➤${COLOR_RESET} $*"; }
ok(){ echo -e "${COLOR_GREEN}✓${COLOR_RESET} $*"; }
warn(){ echo -e "${COLOR_YELLOW}!${COLOR_RESET} $*"; }
err(){ echo -e "${COLOR_RED}✗${COLOR_RESET} $*"; }

require_cmd(){ command -v "$1" >/dev/null 2>&1; }

# Ensure sudo exists (most Arch installs have it)
if ! require_cmd sudo; then
  err "sudo n'est pas disponible. Exécute ce script en root ou installe sudo."
  exit 1
fi
need_multilib=1   # Ce script installe aussi des libs 32-bit utiles aux jeux

if [[ "$need_multilib" -eq 1 ]]; then
  if ! pacman-conf --repo-list | grep -qx "multilib"; then
    err "multilib n'est pas activé. Lance d'abord 02-enable-multilib.sh"
    exit 1
  fi
fi

info "Détection du GPU principal…"
gpu_line="$(lspci -nn | grep -E 'VGA|3D|Display' | head -n1 || true)"
if [[ -z "${gpu_line}" ]]; then
  err "Impossible de détecter le GPU via lspci."
  exit 1
fi
echo "→ ${gpu_line}"

packages=(mesa lib32-mesa vulkan-icd-loader lib32-vulkan-icd-loader libva-mesa-driver lib32-libva-mesa-driver)
vendor="unknown"

shopt -s nocasematch
if [[ "${gpu_line}" == *"NVIDIA"* ]]; then
  vendor="nvidia"
  # Choix dkms pour compatibilité kernels (lts/mainline)
  packages+=(nvidia-dkms nvidia-utils lib32-nvidia-utils nvidia-settings)
  # Optionnel: prise en charge VA-API via nvidia-vaapi-driver (AUR) — non installée ici par défaut
elif [[ "${gpu_line}" == *"AMD"* || "${gpu_line}" == *"ATI"* ]]; then
  vendor="amd"
  packages+=(vulkan-radeon lib32-vulkan-radeon mesa-vdpau lib32-mesa-vdpau)
  # RADV (Mesa) recommandé; on évite amdvlk par défaut
elif [[ "${gpu_line}" == *"Intel"* ]]; then
  vendor="intel"
  packages+=(vulkan-intel lib32-vulkan-intel intel-media-driver)
  # Legacy (optionnel): libva-intel-driver
else
  warn "GPU non reconnu, on installe un socle générique (Mesa/Vulkan)."
fi
shopt -u nocasematch

info "Vendor détecté : ${vendor}"
info "Installation des paquets: ${packages[*]}"
sudo pacman -S --needed --noconfirm "${packages[@]}"

# Steam runtime (optionnel mais utile)
extra_pkgs=(lib32-libunwind lib32-libxcrypt-compat)
info "Installation de quelques libs 32-bit utiles aux jeux: ${extra_pkgs[*]}"
sudo pacman -S --needed --noconfirm "${extra_pkgs[@]}" || true

ok "Drivers et stacks Vulkan installés pour ${vendor}."
