#!/usr/bin/env bash
set -euo pipefail

COLOR_BLUE="\033[1;34m"
COLOR_GREEN="\033[1;32m"
COLOR_YELLOW="\033[1;33m"
COLOR_RED="\033[1;31m"
COLOR_RESET="\033[0m"

info(){ echo -e "${COLOR_BLUE}➤${COLOR_RESET} $*"; }
ok(){ echo -e "${COLOR_GREEN}✓${COLOR_RESET} $*"; }
warn(){ echo -e "${COLOR_YELLOW}!${COLOR_RESET} $*"; }
err(){ echo -e "${COLOR_RED}✗${COLOR_RESET} $*"; }

require_cmd(){ command -v "$1" >/dev/null 2>&1; }

# Ensure sudo exists (most Arch installs have it)
if ! require_cmd sudo; then
  err "sudo n'est pas disponible. Exécute ce script en root ou installe sudo."
  exit 1
fi
# Vérifs préalables
if ! command -v yay >/dev/null 2>&1; then
  err "yay est requis pour installer Discord depuis l'AUR. Lance d'abord 01-install-yay.sh"
  exit 1
fi
if ! pacman-conf --repo-list | grep -qx "multilib"; then
  err "multilib est requis pour Steam. Lance d'abord 02-enable-multilib.sh"
  exit 1
fi

# Steam via pacman
info "Installation de Steam (pacman)…"
sudo pacman -S --needed --noconfirm steam

# Discord via yay (préférence: discord_arch_electron si discord officiel pose souci, fallback automatique)
info "Installation de Discord (AUR) via yay…"
if yay -Ss "^discord_arch_electron$" | grep -q "^aur/discord_arch_electron "; then
  yay -S --needed --noconfirm discord_arch_electron || {
    warn "Échec discord_arch_electron, on tente le paquet discord (AUR/officiel selon dispo)…"
    yay -S --needed --noconfirm discord
  }
else
  yay -S --needed --noconfirm discord || {
    warn "Échec discord (AUR), tentative discord_arch_electron…"
    yay -S --needed --noconfirm discord_arch_electron
  }
fi

# Sober via Flatpak
if ! command -v flatpak >/dev/null 2>&1; then
  info "Flatpak non présent, installation…"
  sudo pacman -S --needed --noconfirm flatpak
fi
if ! flatpak remotes | grep -q "^flathub"; then
  info "Ajout du remote flathub…"
  sudo flatpak remote-add --if-not-exists flathub https://flathub.org/repo/flathub.flatpakrepo
fi

info "Installation de Sober (VinegarHQ) via Flatpak…"
flatpak install -y flathub org.vinegarhq.Sober

ok "Steam, Discord et Sober installés."
